package jadean.test.dean.java.utilities;

import static org.junit.Assert.*;

import jadean.dean.java.utilities.FileUtilities;
import jadean.dean.java.utilities.JavaUtilities;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import org.junit.After;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;


public class FileUtilitiesTest {
	
	@Rule
	public ExpectedException exception = ExpectedException.none();

	@BeforeClass
	public static void setUpBeforeClass() throws Exception {
	}

	@AfterClass
	public static void tearDownAfterClass() throws Exception {
	}

	@Before
	public void setUp() throws Exception {
	}

	@After
	public void tearDown() throws Exception {
	}

	@Test
	public void testFilesInDirs() {
		String rootDir = "test/resources/fileutilities";
		String aDir = "test/resources/fileutilities/a";
		String bDir = "test/resources/fileutilities/b";
		
		File rootFile = new File("test/resources/fileutilities/a.txt").getAbsoluteFile();
		File bFile = new File("test/resources/fileutilities/b/a.txt").getAbsoluteFile();
		
		Collection<File> rootFiles = Arrays.asList(rootFile);
		Collection<File> aFiles = Arrays.asList();
		Collection<File> bFiles = Arrays.asList(bFile);
		
		Assert.assertTrue(JavaUtilities.collectionsEquals(rootFiles, FileUtilities.filesInDirs(Arrays.asList(rootDir))));
		Assert.assertTrue(JavaUtilities.collectionsEquals(rootFiles, FileUtilities.filesInDirs(Arrays.asList(rootDir, aDir))));
		Assert.assertTrue(JavaUtilities.collectionsEquals(aFiles, FileUtilities.filesInDirs(Arrays.asList(aDir))));
		Assert.assertTrue(JavaUtilities.collectionsEquals(bFiles, FileUtilities.filesInDirs(Arrays.asList(bDir))));
		Assert.assertTrue(JavaUtilities.collectionsEquals(bFiles, FileUtilities.filesInDirs(Arrays.asList(bDir, aDir))));
		Assert.assertTrue(JavaUtilities.collectionsEquals(Arrays.asList(rootFile, bFile), FileUtilities.filesInDirs(Arrays.asList(rootDir, bDir))));
		Assert.assertTrue(JavaUtilities.collectionsEquals(Arrays.asList(rootFile, bFile), FileUtilities.filesInDirs(Arrays.asList(rootDir, aDir, bDir))));
	}
	
	@Test
	public void testInnerClassInDir() {
		String dir = "test/resources/fileutilities/c";
		
		Assert.assertTrue(FileUtilities.innerClassInDir(dir, "Inner.class"));
		Assert.assertTrue(FileUtilities.innerClassInDir(dir, "Subinner1.class"));
		Assert.assertTrue(FileUtilities.innerClassInDir(dir, "Subinner2.class"));
		Assert.assertTrue(FileUtilities.innerClassInDir(dir, "Subsub.class"));
		Assert.assertFalse(FileUtilities.innerClassInDir(dir, "Main.class"));
	}	

	@Test
	public void testAllFilesInDirs() {
		String aDir = "test/resources/fileutilities/a";
		String bDir = "test/resources/fileutilities/b";
		String dDir = "test/resources/fileutilities/d";
		String eDir = "test/resources/fileutilities/e";
		String fDir = "test/resources/fileutilities/f";
		
		List<File> aFiles = FileUtilities.allFilesInDirs(Arrays.asList(aDir, bDir, dDir, eDir, fDir));
		List<File> eFiles = Arrays.asList(
				new File("test/resources/fileutilities/b/a.txt").getAbsoluteFile(),
				new File("test/resources/fileutilities/d/a.txt").getAbsoluteFile(),
				new File("test/resources/fileutilities/d/a/a.txt").getAbsoluteFile(),
				new File("test/resources/fileutilities/e/a.txt").getAbsoluteFile(),
				new File("test/resources/fileutilities/f/a.txt").getAbsoluteFile(),
				new File("test/resources/fileutilities/f/a/a.txt").getAbsoluteFile(),
				new File("test/resources/fileutilities/f/a/a/a.txt").getAbsoluteFile(),
				new File("test/resources/fileutilities/f/a/a/a/a.txt").getAbsoluteFile()
				);
		
		Assert.assertTrue(JavaUtilities.collectionsEquals(aFiles, eFiles));
	}

	@Test
	public void testFindInDirs() {
		String aDir = "test/resources/fileutilities/a";
		String bDir = "test/resources/fileutilities/b";
		String dDir = "test/resources/fileutilities/d";
		String eDir = "test/resources/fileutilities/e";
		String fDir = "test/resources/fileutilities/f";
		
		String fileFound = "a/a/a/a.txt";
		String fileNotFound = "b/a.txt";
		
		String eFileName = new File(fDir, fileFound).getAbsolutePath();
		
		Assert.assertNull(FileUtilities.findInDirs(new ArrayList<String>(), fileNotFound));
		Assert.assertNull(FileUtilities.findInDirs(Arrays.asList(aDir, bDir, dDir, eDir, fDir), fileNotFound));
		Assert.assertEquals(eFileName, FileUtilities.findInDirs(Arrays.asList(aDir, bDir, dDir, eDir, fDir), fileFound));
	}

	@Test
	public void testGetClassName() {
		String iDir = new File("test/resources/fileutilities/i").getAbsolutePath();
		String jDir = new File("test/resources/fileutilities/j").getAbsolutePath();
		String aDir = new File("test/resources/fileutilities/a").getAbsolutePath();
		
		String aMain = "a.Main";
		String aaMain = "a.a.Main";
		String aaaSource = "a.a.a.Source";
		String main = "Main";
		String fake = "Fake";
		String aFake = "a.Fake";
		
		File iaMainFile = new File(iDir, aMain.replace('.', File.separatorChar)).getAbsoluteFile();
		File jaMainFile = new File(jDir, aMain.replace('.', File.separatorChar)).getAbsoluteFile();
		File iaaMainFile = new File(iDir, aaMain.replace('.', File.separatorChar)).getAbsoluteFile();
		File jaaMainFile = new File(jDir, aaMain.replace('.', File.separatorChar)).getAbsoluteFile();
		File iaaaSourceFile = new File(iDir, aaaSource.replace('.', File.separatorChar)).getAbsoluteFile();
		File jMainFile = new File(jDir, main.replace('.', File.separatorChar)).getAbsoluteFile();
		File iFake = new File(aDir, fake.replace('.', File.separatorChar)).getAbsoluteFile();
		File iaFake = new File(aDir, aFake.replace('.', File.separatorChar)).getAbsoluteFile();
		
		Assert.assertEquals(aMain, FileUtilities.getClassName(iaMainFile, Arrays.asList(iDir)));
		Assert.assertEquals(aMain, FileUtilities.getClassName(jaMainFile, Arrays.asList(jDir)));
		Assert.assertEquals(aaMain, FileUtilities.getClassName(iaaMainFile, Arrays.asList(iDir)));
		Assert.assertEquals(aaMain, FileUtilities.getClassName(jaaMainFile, Arrays.asList(jDir)));
		Assert.assertEquals(aaaSource, FileUtilities.getClassName(iaaaSourceFile, Arrays.asList(iDir, jDir)));
		Assert.assertEquals(main, FileUtilities.getClassName(jMainFile, Arrays.asList(iDir, jDir)));
		Assert.assertNull(FileUtilities.getClassName(iFake, Arrays.asList(iDir, jDir)));
		Assert.assertNull(FileUtilities.getClassName(iaFake, Arrays.asList(iDir, jDir)));
	}
	
	@Test
	public void testRenameTo() {
		String oldFileName = new File("test/resources/fileutilities/rename/a.txt").getAbsolutePath();
		String newFileName = new File("test/resources/fileutilities/rename/b.txt").getAbsolutePath();
		
		File oldFile = new File(oldFileName);
		if (!oldFile.exists()) {
			try {
				oldFile.createNewFile();
			} catch (IOException e) {
				fail("Could not create new file for testing");
			}
		}
		
		try {
			FileUtilities.renameTo(oldFileName, newFileName);
			Assert.assertTrue(new File(newFileName).exists());
			Assert.assertFalse(new File(oldFileName).exists());
			new File(newFileName).delete();
			new File(oldFileName).delete();
		}
		catch (Exception e) {
			fail("IOException thrown: " + e.getMessage());
		}
		
		
		String fakeFileName = new File("test/resources/fileutilities/rename/fake.txt").getAbsolutePath();
		try {
			FileUtilities.renameTo(fakeFileName, newFileName);
			fail("IOException was not thrown even though file did not exist");
		}
		catch (IOException e) {}
		
		
		oldFile = new File(oldFileName);
		File newFile = new File(newFileName);
		if (!oldFile.exists()) {
			try {
				oldFile.createNewFile();
			} catch (IOException e) {
				fail("Could not create new file for testing");
			}
		}
		if (!newFile.exists()) {
			try {
				newFile.createNewFile();
			} catch (IOException e) {
				fail("Could not create new file for testing");
			}
		}		
		BufferedReader br = null;
		try {
			br = new BufferedReader(new FileReader(newFile));
			br.read();
		}
		catch (Exception e) {
			fail("Could not open used file for reading");
		}
		
		try {
			FileUtilities.renameTo(oldFileName, newFileName);
			fail("IOException was not thrown even though file was not closed");
		}
		catch (Exception e) {}
		
		try {
			br.close();
			FileUtilities.deleteFile(oldFileName);
		}
		catch (Exception e) {
			fail("Could not delete used file for testing");
		}
		
		try {
			FileUtilities.deleteFile(newFileName);
		}
		catch (Exception e) {}
		
	}

	@Test
	public void testCopyfile() {
		fail("Not yet implemented");
	}
	
	@Test
	public void testDeletefile() {
		fail("Not yet implemented");
	}

	@Test
	public void testInputStreamToString() {
		fail("Not yet implemented");
	}
}